<?php

namespace App\Controllers;

use App\Models\PpdbModel;
use App\Models\PengaturanwebModel;
use App\Models\ArtikelModel; // karena tabel berita kamu namanya artikel
use App\Models\FasilitasModel;
use App\Models\SiswaModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Dompdf\Dompdf;

class Ppdb extends BaseController
{
    protected $ppdbModel;

    public function __construct()
    {
        $this->ppdbModel = new PpdbModel();
    }

    // ---------- FORM PUBLIK ----------
    public function index()
    {
        $settingModel   = new PengaturanwebModel();
        $artikelModel   = new ArtikelModel();
        $fasilitasModel = new FasilitasModel();

        $setting   = $settingModel->first();
        $berita    = $artikelModel->orderBy('created_at', 'DESC')->limit(5)->findAll();
        $fasilitas = $fasilitasModel->findAll();

        return view('ppdb_form', [
            'title'       => 'Formulir Pendaftaran PPDB',
            'namaSekolah' => $setting['nama_sekolah'] ?? 'Sekolah',
            'setting'     => $setting,
            'berita'      => $berita,
            'fasilitas'   => $fasilitas
        ]);
    }

    // Simpan data pendaftaran dari form publik
    public function save()
    {
        // Debug: Log data yang diterima
        log_message('debug', 'PPDB Save Data: ' . json_encode($this->request->getPost()));
        
        // Validasi basic
        if (!$this->request->getPost('nama_lengkap')) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Nama lengkap harus diisi!');
        }
        
        if (!$this->request->getPost('jenis_kelamin')) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Jenis kelamin harus dipilih!');
        }

        // Validasi file upload
        $buktiTf = $this->request->getFile('bukti_tf');
        $aktaKelahiran = $this->request->getFile('akta_kelahiran');
        $kartuKeluarga = $this->request->getFile('kartu_keluarga');

        if (!$buktiTf || !$buktiTf->isValid()) {
            return redirect()->to(base_url('ppdb'))->with('error', 'File bukti transfer harus diupload!');
        }

        if (!$aktaKelahiran || !$aktaKelahiran->isValid()) {
            return redirect()->to(base_url('ppdb'))->with('error', 'File akta kelahiran harus diupload!');
        }

        if (!$kartuKeluarga || !$kartuKeluarga->isValid()) {
            return redirect()->to(base_url('ppdb'))->with('error', 'File kartu keluarga harus diupload!');
        }

        // Validasi ukuran file (max 2MB)
        $maxSize = 2048; // 2MB dalam KB
        if ($buktiTf->getSizeByUnit('kb') > $maxSize) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Ukuran file bukti transfer terlalu besar (max 2MB)!');
        }

        if ($aktaKelahiran->getSizeByUnit('kb') > $maxSize) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Ukuran file akta kelahiran terlalu besar (max 2MB)!');
        }

        if ($kartuKeluarga->getSizeByUnit('kb') > $maxSize) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Ukuran file kartu keluarga terlalu besar (max 2MB)!');
        }

        // Validasi tipe file
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];
        if (!in_array($buktiTf->getMimeType(), $allowedTypes)) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Format file bukti transfer tidak valid! Gunakan JPG, PNG, atau PDF.');
        }

        if (!in_array($aktaKelahiran->getMimeType(), $allowedTypes)) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Format file akta kelahiran tidak valid! Gunakan JPG, PNG, atau PDF.');
        }

        if (!in_array($kartuKeluarga->getMimeType(), $allowedTypes)) {
            return redirect()->to(base_url('ppdb'))->with('error', 'Format file kartu keluarga tidak valid! Gunakan JPG, PNG, atau PDF.');
        }
        
        try {
            // Generate nama file unik
            $timestamp = time();
            $buktiTfName = 'bukti_tf_' . $timestamp . '_' . $buktiTf->getRandomName();
            $aktaName = 'akta_' . $timestamp . '_' . $aktaKelahiran->getRandomName();
            $kkName = 'kk_' . $timestamp . '_' . $kartuKeluarga->getRandomName();

            // Upload files
            $uploadPath = FCPATH . 'uploads/ppdb/';
            if (!$buktiTf->move($uploadPath, $buktiTfName)) {
                return redirect()->to(base_url('ppdb'))->with('error', 'Gagal mengupload bukti transfer!');
            }

            if (!$aktaKelahiran->move($uploadPath, $aktaName)) {
                return redirect()->to(base_url('ppdb'))->with('error', 'Gagal mengupload akta kelahiran!');
            }

            if (!$kartuKeluarga->move($uploadPath, $kkName)) {
                return redirect()->to(base_url('ppdb'))->with('error', 'Gagal mengupload kartu keluarga!');
            }

            // Insert data ke database
            $this->ppdbModel->insert([
                'nama_lengkap'     => $this->request->getPost('nama_lengkap'),
                'nama_panggilan'   => $this->request->getPost('nama_panggilan'),
                'jenis_kelamin'    => $this->request->getPost('jenis_kelamin'),
                'tempat_tgl_lahir' => $this->request->getPost('tempat_tgl_lahir'),
                'agama'            => $this->request->getPost('agama'),
                'anak_nomor_ke'    => $this->request->getPost('anak_nomor_ke'),
                'banyak_saudara'   => $this->request->getPost('banyak_saudara'),
                'bahasa_sehari'    => $this->request->getPost('bahasa_sehari'),
                'berat_badan'      => $this->request->getPost('berat_badan'),
                'tinggi_badan'     => $this->request->getPost('tinggi_badan'),
                'umur_saat_daftar' => $this->request->getPost('umur_saat_daftar'),
                'nama_ayah'        => $this->request->getPost('nama_ayah'),
                'nama_ibu'         => $this->request->getPost('nama_ibu'),
                'pendidikan_ayah'  => $this->request->getPost('pendidikan_ayah'),
                'pendidikan_ibu'   => $this->request->getPost('pendidikan_ibu'),
                'pekerjaan_ayah'   => $this->request->getPost('pekerjaan_ayah'),
                'pekerjaan_ibu'    => $this->request->getPost('pekerjaan_ibu'),
                'nama_wali'        => $this->request->getPost('nama_wali'),
                'pendidikan_wali'  => $this->request->getPost('pendidikan_wali'),
                'hubungan_wali'    => $this->request->getPost('hubungan_wali'),
                'pekerjaan_wali'   => $this->request->getPost('pekerjaan_wali'),
                'alamat_wali'      => $this->request->getPost('alamat_wali'),
                'asal_mula'        => $this->request->getPost('asal_mula'),
                'bukti_tf'         => $buktiTfName,
                'akta_kelahiran'   => $aktaName,
                'kartu_keluarga'   => $kkName,
                'no_hp'            => $this->request->getPost('no_hp'),
                'created_at'       => date('Y-m-d H:i:s'),
                'updated_at'       => date('Y-m-d H:i:s')
            ]);

            return redirect()->to(base_url('ppdb'))->with('success', 'Pendaftaran berhasil dikirim! Data Anda akan diverifikasi oleh admin.');
            
        } catch (\Exception $e) {
            log_message('error', 'PPDB Save Error: ' . $e->getMessage());
            return redirect()->to(base_url('ppdb'))->with('error', 'Terjadi kesalahan saat menyimpan data. Silakan coba lagi. Error: ' . $e->getMessage());
        }
    }

    // ---------- ADMIN ----------
    // List semua pendaftar
    public function adminIndex()
    {
        // Load PengaturanwebModel untuk mendapatkan nama sekolah
        $pengaturanModel = new \App\Models\PengaturanwebModel();
        $setting = $pengaturanModel->first();
        
        $data = [
            'title' => 'Data Pendaftar PPDB',
            'pendaftar' => $this->ppdbModel->orderBy('created_at', 'DESC')->findAll(),
            'setting' => $setting
        ];
        return view('admin/ppdb/index', $data);
    }

    // Halaman form pendaftaran (jika admin mau input manual)
    public function pendaftaran()
    {
        $data = [
            'title' => 'Form Pendaftaran PPDB'
        ];
        return view('admin/ppdb/pendaftaran', $data);
    }

    // Detail data siswa
    public function detail($id)
    {
        $siswa = $this->ppdbModel->find($id);

        if (!$siswa) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound("Data tidak ditemukan");
        }

        $data = [
            'title' => 'Detail Pendaftar PPDB',
            'siswa' => $siswa
        ];

        return view('admin/ppdb/detail', $data);
    }

    // Hapus data
    public function delete($id)
    {
        $this->ppdbModel->delete($id);
        return redirect()->to(base_url('admin/ppdb'))->with('success', 'Data berhasil dihapus!');
    }

    // Edit data (form)
    public function edit($id)
    {
        $siswa = $this->ppdbModel->find($id);

        if (!$siswa) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound("Data tidak ditemukan");
        }

        $data = [
            'title' => 'Edit Data Pendaftar PPDB',
            'siswa' => $siswa
        ];

        return view('admin/ppdb/edit', $data);
    }

    // Update data
    public function update($id)
    {
        // Debug: Log data yang diterima
        log_message('info', 'PPDB Update called for ID: ' . $id);
        log_message('info', 'Alamat Wali from POST: ' . ($this->request->getPost('alamat_wali') ?? 'NULL'));
        
        try {
            // Get existing data
            $existingData = $this->ppdbModel->find($id);
            if (!$existingData) {
                return redirect()->to(base_url('admin/ppdb'))->with('error', 'Data tidak ditemukan!');
            }
            
            $data = [
                'nama_lengkap'     => $this->request->getPost('nama_lengkap'),
                'nama_panggilan'   => $this->request->getPost('nama_panggilan'),
                'jenis_kelamin'    => $this->request->getPost('jenis_kelamin'),
                'tempat_tgl_lahir' => $this->request->getPost('tempat_tgl_lahir'),
                'agama'            => $this->request->getPost('agama'),
                'anak_nomor_ke'    => $this->request->getPost('anak_nomor_ke'),
                'banyak_saudara'   => $this->request->getPost('banyak_saudara'),
                'berat_badan'      => $this->request->getPost('berat_badan'),
                'tinggi_badan'     => $this->request->getPost('tinggi_badan'),
                'umur_saat_daftar' => $this->request->getPost('umur_saat_daftar'),
                'bahasa_sehari'    => $this->request->getPost('bahasa_sehari'),
                'nama_ayah'        => $this->request->getPost('nama_ayah'),
                'nama_ibu'         => $this->request->getPost('nama_ibu'),
                'pendidikan_ayah'  => $this->request->getPost('pendidikan_ayah'),
                'pendidikan_ibu'   => $this->request->getPost('pendidikan_ibu'),
                'pekerjaan_ayah'   => $this->request->getPost('pekerjaan_ayah'),
                'pekerjaan_ibu'    => $this->request->getPost('pekerjaan_ibu'),
                'nama_wali'        => $this->request->getPost('nama_wali'),
                'hubungan_wali'    => $this->request->getPost('hubungan_wali'),
                'pendidikan_wali'  => $this->request->getPost('pendidikan_wali'),
                'pekerjaan_wali'   => $this->request->getPost('pekerjaan_wali'),
                'alamat_wali'      => $this->request->getPost('alamat_wali'),
                'asal_mula'        => $this->request->getPost('asal_mula'),
                'status'           => $this->request->getPost('status'),
                'no_hp'            => $this->request->getPost('no_hp'),
                'updated_at'       => date('Y-m-d H:i:s')
            ];

            // Handle file uploads
            $uploadPath = FCPATH . 'uploads/ppdb/';
            $maxSize = 2048; // 2MB dalam KB
            $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];

            // Handle Bukti Transfer
            $buktiTf = $this->request->getFile('bukti_tf');
            if ($buktiTf && $buktiTf->isValid() && !$buktiTf->hasMoved()) {
                // Validasi file
                if ($buktiTf->getSizeByUnit('kb') > $maxSize) {
                    return redirect()->back()->with('error', 'Ukuran file bukti transfer terlalu besar (max 2MB)!')->withInput();
                }
                if (!in_array($buktiTf->getMimeType(), $allowedTypes)) {
                    return redirect()->back()->with('error', 'Format file bukti transfer tidak valid!')->withInput();
                }

                // Hapus file lama jika ada
                if (!empty($existingData['bukti_tf']) && file_exists($uploadPath . $existingData['bukti_tf'])) {
                    unlink($uploadPath . $existingData['bukti_tf']);
                }

                // Upload file baru  
                $buktiTfName = 'bukti_tf_' . time() . '_' . $buktiTf->getRandomName();
                if ($buktiTf->move($uploadPath, $buktiTfName)) {
                    $data['bukti_tf'] = $buktiTfName;
                }
            }

            // Handle Akta Kelahiran
            $aktaKelahiran = $this->request->getFile('akta_kelahiran');
            if ($aktaKelahiran && $aktaKelahiran->isValid() && !$aktaKelahiran->hasMoved()) {
                // Validasi file
                if ($aktaKelahiran->getSizeByUnit('kb') > $maxSize) {
                    return redirect()->back()->with('error', 'Ukuran file akta kelahiran terlalu besar (max 2MB)!')->withInput();
                }
                if (!in_array($aktaKelahiran->getMimeType(), $allowedTypes)) {
                    return redirect()->back()->with('error', 'Format file akta kelahiran tidak valid!')->withInput();
                }

                // Hapus file lama jika ada
                if (!empty($existingData['akta_kelahiran']) && file_exists($uploadPath . $existingData['akta_kelahiran'])) {
                    unlink($uploadPath . $existingData['akta_kelahiran']);
                }

                // Upload file baru
                $aktaName = 'akta_' . time() . '_' . $aktaKelahiran->getRandomName();
                if ($aktaKelahiran->move($uploadPath, $aktaName)) {
                    $data['akta_kelahiran'] = $aktaName;
                }
            }

            // Handle Kartu Keluarga
            $kartuKeluarga = $this->request->getFile('kartu_keluarga');
            if ($kartuKeluarga && $kartuKeluarga->isValid() && !$kartuKeluarga->hasMoved()) {
                // Validasi file
                if ($kartuKeluarga->getSizeByUnit('kb') > $maxSize) {
                    return redirect()->back()->with('error', 'Ukuran file kartu keluarga terlalu besar (max 2MB)!')->withInput();
                }
                if (!in_array($kartuKeluarga->getMimeType(), $allowedTypes)) {
                    return redirect()->back()->with('error', 'Format file kartu keluarga tidak valid!')->withInput();
                }

                // Hapus file lama jika ada
                if (!empty($existingData['kartu_keluarga']) && file_exists($uploadPath . $existingData['kartu_keluarga'])) {
                    unlink($uploadPath . $existingData['kartu_keluarga']);
                }

                // Upload file baru
                $kkName = 'kk_' . time() . '_' . $kartuKeluarga->getRandomName();
                if ($kartuKeluarga->move($uploadPath, $kkName)) {
                    $data['kartu_keluarga'] = $kkName;
                }
            }
            
            log_message('info', 'Data to update: ' . json_encode($data));
            
            $result = $this->ppdbModel->update($id, $data);
            log_message('info', 'Update result: ' . ($result ? 'SUCCESS' : 'FAILED'));

            return redirect()->to(base_url('admin/ppdb'))->with('success', 'Data berhasil diperbarui!');
            
        } catch (\Exception $e) {
            log_message('error', 'PPDB Update Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Terjadi kesalahan saat mengupdate data: ' . $e->getMessage())->withInput();
        }
    }

    // Approve pendaftar dan pindah ke tabel siswa
    public function approve($id)
    {
        $pendaftar = $this->ppdbModel->find($id);
        
        if (!$pendaftar) {
            return redirect()->to(base_url('admin/ppdb'))->with('error', 'Data tidak ditemukan!');
        }

        // Update status di tabel ppdb
        $this->ppdbModel->update($id, [
            'status' => 'diterima',
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        // Pindahkan ke tabel siswa
        $siswaModel = new \App\Models\SiswaModel();
        
        $siswaModel->insert([
            'nisn' => $this->generateNISN(),
            'nama_siswa' => $pendaftar['nama_lengkap'],
            'tmp_lahir' => $this->extractTempat($pendaftar['tempat_tgl_lahir']),
            'tgl_lahir' => $this->extractTanggal($pendaftar['tempat_tgl_lahir']),
            'jk_siswa' => $pendaftar['jenis_kelamin'] == 'Laki-laki' ? '1' : '0',
            'no_telp' => $pendaftar['no_hp'],
            'alamat' => $pendaftar['alamat_wali'],
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        return redirect()->to(base_url('admin/ppdb'))->with('success', 'Pendaftar berhasil diterima dan dipindahkan ke data siswa!');
    }

    // Reject pendaftar
    public function reject($id)
    {
        $this->ppdbModel->update($id, [
            'status' => 'ditolak',
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        return redirect()->to(base_url('admin/ppdb'))->with('success', 'Pendaftar ditolak!');
    }

    // Helper untuk generate NISN
    private function generateNISN()
    {
        $year = date('Y');
        $siswaModel = new \App\Models\SiswaModel();
        $lastSiswa = $siswaModel->orderBy('id', 'DESC')->first();
        $lastNumber = $lastSiswa ? (int)substr($lastSiswa['nisn'], -4) : 0;
        $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        
        return $year . $newNumber;
    }

    // Helper untuk extract tempat lahir
    private function extractTempat($tempat_tgl_lahir)
    {
        if (strpos($tempat_tgl_lahir, ',') !== false) {
            return trim(explode(',', $tempat_tgl_lahir)[0]);
        }
        return $tempat_tgl_lahir;
    }

    // Helper untuk extract tanggal lahir
    private function extractTanggal($tempat_tgl_lahir)
    {
        if (strpos($tempat_tgl_lahir, ',') !== false) {
            $tanggal = trim(explode(',', $tempat_tgl_lahir)[1]);
            // Convert format tanggal jika perlu
            return date('Y-m-d', strtotime($tanggal));
        }
        return date('Y-m-d');
    }
}
